﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_DisabiltyRatingScale : BaseTBIInstrumentPage {

    #region Variable Declarations

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    protected int _surveyTypeId = 12; // Database Id for Disabilty Rating Scale

    SURVEY _survey;   // the root object in the Survey set
    SURVEY_RESULT _surveyResult; // the object representing the actual data in the Survey

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();

    #endregion

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_DisabiltyRatingScale() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }

    #region Protected Methods

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e) {
        TBIServiceTO _tbiServiceTo;

        // confirm instrument is active - if not, a redirect will occur.
        this.CheckActiveStatus(_surveyTypeId);

        try {
            _tbiServiceTo = (TBIServiceTO)Session["TBIServiceTO"];
            _tbiInstrumentManager.TbiServiceTo = _tbiServiceTo;

            string instTitle = _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > " + instTitle;
            LabelInstrumentTitle.Text = instTitle;
            Page.Title = "TBI Instruments - " + instTitle;

            if (_tbiInstrumentManager.TbiServiceTo != null) {
                Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
                labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSnum.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);   
            }
        } catch (Exception ex) {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "DisabiltyRatingScale.aspx", "100027", "SESSIOENDED");
            Response.Redirect("Warning.aspx?err=2", false);
        }

        if (!IsPostBack) {
            if (instrumentType.Status != null) {
                _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
            }

            string _snum = "";
            if (_tbiInstrumentManager.TbiServiceTo != null) {
                _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
            }

            if (!string.IsNullOrEmpty(_snum))
            {
                string patICN = patientManager.GetPatientICNBySnum(_snum);

                if (!string.IsNullOrEmpty(patICN)) {
                    ViewState["PatientICN"] = patICN;
                    RefreshSurvey(patICN);
                } else {
                    throw new ArgumentException("Patient with this SSN does not exist.");
                }

                if (_survey != null && !_survey.isNew) {
                    SetUpSurvey(_survey);
                }
                string referrer = "";
                if (Request.UrlReferrer != null) {
                    referrer = Request.UrlReferrer.PathAndQuery;
                }
                ViewState["referrer"] = referrer;
            }

            //Check if the request is to start a new item 
            //a new item page request is signified when the querystring parameter id equals zero or is not present
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("DisabilityRatingScale.Page_Load(): WarningRedirect: _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            } else {
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSnum.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
                //LabelFacility.Text = _tbiInstrumentManager.TbiServiceTo.InstitutionName;
            }
        }
        QuestionList = GetAllQuestionsForSurvey();

        LabelDate1.Text = GetQuestionText("DTE");
        LabelForTypeOfEncounter.Text = GetQuestionText("TYP");

        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        LabelForQA.Text = "A. " + GetQuestionText("A");
        LabelForQB.Text = "B. " + GetQuestionText("B");
        LabelForQC.Text = "C. " + GetQuestionText("C");
        LabelForQD.Text = "D. " + GetQuestionText("D");
        LabelForQE.Text = "E. " + GetQuestionText("E");
        LabelForQF.Text = "F. " + GetQuestionText("F");
        LabelForQG.Text = "G. " + GetQuestionText("G");
        LabelForQH.Text = "H. " + GetQuestionText("H");

        dpDate1.MaxDate = DateTime.Now;
    }

    protected override void LoadViewState(object savedState) {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState() {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower()) {
                return ques.QUESTION_TEXT;
            }
        }

        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower()) {
                return ques.ID;
            }
        }

        return 0;
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        SetUpSurvey(_survey);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);

        if (status > 0) {
            Session["survey"] = _survey;
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        } else {
            SetUpSurvey(_survey); //Follow Save Draft logic 
        }
    }

    /// <summary>
    /// Inactive timer firing.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void InActiveTimer_Tick(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";

        Page.Validate();

        if (Page.IsValid) {
            SaveCompleteSurvey(ref status);
            //SetUpSurvey(_survey);
            Response.Redirect("DisabiltyRatingScale.aspx", false);
        } else {
            SaveFailMessage = "Auto-Save Validation Failed. Please complete required fields in the survey and press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        }
    }

    /// <summary>
    /// Cancel button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
        //this.ClientScript.RegisterStartupScript(typeof(Page), "closePage", "<script type='text/javascript'> window.open('','_self','');window.close(); </script>");
    }

    /// <summary>
    /// Hide or Show instructions
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void lnkBtnToggleInstructions_OnClick(object sender, EventArgs e) {
        if (PanelInstructions.Visible) {
            PanelInstructions.Visible = false;
            lnkBtnToggleInstructions.Text = "Click for Instructions";
        } else {
            PanelInstructions.Visible = true;
            lnkBtnToggleInstructions.Text = "Hide Instructions";
        }
    }

    #endregion

    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status) {
        Page.Validate();
        if (Page.IsValid) {
            string patientICN = String.Empty;

            status = 0;

            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null) {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("DisabilityRatingScale.SaveCompletedSurvey(): WarningRedirect: String.IsNullOrEmpty(patientICN.Trim()");
                    //Lost state. Redirect to message that need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("DisabilityRatingScale.SaveCompletedSurvey(): WarningRedirect: ViewState[PatientICN] != null");
                //Lost state. Redirect to message that need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null) {
                if (_survey.isNew) {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("DisabilityRatingScale.SaveCompletedSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    } else {
                        _survey.SURVEY_STATUS = "inprocess";

                        _survey.SetAsInsertOnSubmit();

                        _survey.patient_ICN = patientICN;

                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);

                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);

                //Fortify errors on this line (this.Master as RegistryInstrumentMaster).SetDirtyFlag(false);
                ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);

                RefreshSurvey(patientICN);
                SuppressSaveMessage = false;
            }
        }
    }

    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        List<STD_QUESTION> questions = GetAllQuestionsForSurvey();

        //Delete all the survey results from database if there are any.
        SURVEY surveyWithResults = _tbiInstrumentManager.GetSurveyWithAllAnswers(survey.SURVEYS_ID);
        _tbiInstrumentManager.DeleteAllSurveyResults(surveyWithResults);

        // Question DTE
        _surveyResult = new SURVEY_RESULT();

        if (dpDate1.SelectedDate != null) {
            STD_QUESTION questionDTE = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "DTE");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question DTE choice
                _surveyResult.STD_QUESTION_CHOICE_ID = questionDTE.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = ((DateTime)dpDate1.SelectedDate).ToShortDateString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question DTE");
            }
        }

        // Question TOE
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question TOE choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblTOE.SelectedValue);
            _surveyResult.RESULT_TEXT = rblTOE.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question TOE");
        }

        // Question DX
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(ctrlDiagnosis.GetSelectedId());
            _surveyResult.RESULT_TEXT = ctrlDiagnosis.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question DX");
        }

        // Question A
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question A choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQA.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQA.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question A");
        }

        // Question B
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question B choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQB.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQB.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question B");
        }

        // Question C
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question C choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQC.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQC.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question C");
        }

        // Question D
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question D choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQD.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQD.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question D");
        }

        // Question E
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question E choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQE.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQE.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question E");
        }

        // Question F
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question F choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQF.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQF.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question F");
        }

        // Question G
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question G choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQG.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQG.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question G");
        }

        // Question H
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question H choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQH.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQH.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "DisabiltyRatingScale.aspx", "100027", "SaveSurvey() - Question H");
        }
    }// END SaveSurvey

    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey) {
        try {
            foreach (var sResult in survey.SURVEY_RESULTs) {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER) {
                    case "DTE":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            DateTime temp;
                            if (DateTime.TryParse(sResult.RESULT_TEXT, out temp)) {
                                dpDate1.SelectedDate = temp;
                            }
                        }

                        break;
                    case "TYP":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblTOE.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblTOE.SelectedIndex = -1;
                        }

                        break;
                    case "DX":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            ctrlDiagnosis.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

                            if (sResult.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other")) {
                                ctrlDiagnosis.SetOtherDescriptionText(sResult.RESULT_TEXT);
                            }
                        } else {
                            ctrlDiagnosis.SelectedChoiceId = null;
                        }
                        break;
                    case "A":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQA.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQA.SelectedIndex = -1;
                        }

                        break;
                    case "B":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQB.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQB.SelectedIndex = -1;
                        }

                        break;
                    case "C":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQC.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQC.SelectedIndex = -1;
                        }

                        break;
                    case "D":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQD.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQD.SelectedIndex = -1;
                        }

                        break;
                    case "E":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQE.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQE.SelectedIndex = -1;
                        }

                        break;
                    case "F":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQF.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQF.SelectedIndex = -1;
                        }

                        break;
                    case "G":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQG.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQG.SelectedIndex = -1;
                        }

                        break;
                    case "H":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQH.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQH.SelectedIndex = -1;
                        }

                        break;
                    default:
                        break;
                }
            }
        } catch (Exception ex) {
            Exception _tbilogerror = HandleTBIException(ex, "DisabiltyRatingScale.aspx", "100027", "SetUpSurvey()");
        }
    }// END SetUpSurvey

    #endregion
}
